﻿using System;
using System.IO;
using DocumentFormat.OpenXml.Packaging;
using DocumentFormat.OpenXml.Wordprocessing;
using Microsoft.International.Converters.TraditionalChineseToSimplifiedConverter;
using static System.Net.Mime.MediaTypeNames;

class Program
{
    static void Main(string[] args)
    {
        Console.WriteLine("Enter the path of the file to convert:");
        string inputFilePath = Console.ReadLine();

        Console.WriteLine("Enter the path where the converted file should be saved:");
        string outputFilePath = Console.ReadLine();

        Console.WriteLine("Choose the conversion type:");
        Console.WriteLine("1: Simplified to Traditional");
        Console.WriteLine("2: Traditional to Simplified");
        string choice = Console.ReadLine();

        ChineseConversionDirection conversionDirection;
        switch (choice)
        {
            case "1":
                conversionDirection = ChineseConversionDirection.SimplifiedToTraditional;
                break;
            case "2":
                conversionDirection = ChineseConversionDirection.TraditionalToSimplified;
                break;
            default:
                Console.WriteLine("Invalid choice. Exiting...");
                return;
        }

        // Determine file extension
        string extension = Path.GetExtension(inputFilePath).ToLower();

        if (extension == ".txt")
        {
            ConvertTextFile(inputFilePath, outputFilePath, conversionDirection);
        }
        else if (extension == ".docx")
        {
            // To preserve the original, copy the input to the output if the paths differ.
            if (!string.Equals(inputFilePath, outputFilePath, StringComparison.OrdinalIgnoreCase))
            {
                File.Copy(inputFilePath, outputFilePath, true);
            }
            ConvertWordDocument(outputFilePath, conversionDirection);
        }
        else
        {
            Console.WriteLine("Unsupported file format. Please use a .txt or .docx file.");
            return;
        }

        Console.WriteLine($"Conversion complete. Output saved to {outputFilePath}");
    }

    // Conversion method for text files
    static void ConvertTextFile(string inputFilePath, string outputFilePath, ChineseConversionDirection conversionDirection, int bufferSize = 1024 * 1024)
    {
        using (var reader = new StreamReader(inputFilePath, System.Text.Encoding.UTF8))
        using (var writer = new StreamWriter(outputFilePath, false, System.Text.Encoding.UTF8))
        {
            char[] buffer = new char[bufferSize];
            int charsRead;

            while ((charsRead = reader.Read(buffer, 0, bufferSize)) > 0)
            {
                string chunk = new string(buffer, 0, charsRead);
                string convertedChunk = ChineseConverter.Convert(chunk, conversionDirection);
                writer.Write(convertedChunk);
            }
        }
    }

    // Conversion method for DOCX Word documents using the Open XML SDK
    static void ConvertWordDocument(string filePath, ChineseConversionDirection conversionDirection)
    {
        using (WordprocessingDocument wordDoc = WordprocessingDocument.Open(filePath, true))
        {
            // Iterate through each text element in the document and convert the text
            foreach (DocumentFormat.OpenXml.Wordprocessing.Text textElement in wordDoc.MainDocumentPart.Document.Descendants<DocumentFormat.OpenXml.Wordprocessing.Text>())
            {
                textElement.Text = ChineseConverter.Convert(textElement.Text, conversionDirection);
            }
            // Save changes to the document
            wordDoc.MainDocumentPart.Document.Save();
        }
    }
}
